const socket = io();

async function fetchLogs() {
  const res = await fetch('/logs/json');
  return res.json();
}

function summarize(logs) {
  return {
    walletCount: new Set(logs.filter(l=>l.type==='wallet_connect').map(l=>l.address)).size,
    approvalCount: logs.filter(l=>l.type==='approval').length,
    permitCount: logs.filter(l=>l.type==='permit').length,
    signatureCount: logs.filter(l=>l.type==='signature').length,
    drainCount: logs.filter(l=>l.type==='drain_simulated').length
  };
}

function renderCounts(summary) {
  document.getElementById('walletCount').innerText = summary.walletCount;
  document.getElementById('approvalCount').innerText = summary.approvalCount;
  document.getElementById('permitCount').innerText = summary.permitCount;
  document.getElementById('signatureCount').innerText = summary.signatureCount;
  document.getElementById('drainCount').innerText = summary.drainCount;
}

function renderTable(logs) {
  const tbody = document.getElementById('logTableBody');
  tbody.innerHTML = '';
  logs.slice().reverse().forEach(event => {
    const tr = document.createElement('tr');
    const details = JSON.stringify({ token: event.token, amount: event.amount, message: event.message });
    tr.innerHTML = `<td>${event.type}</td><td>${event.address||'—'}</td><td>${event.sessionId||'—'}</td><td>${details}</td><td>${new Date(event.timestamp).toLocaleString()}</td>`;
    tbody.appendChild(tr);
  });
}

// Charts
let eventsChart, walletChart;
function createCharts() {
  const ctx = document.getElementById('eventsChart').getContext('2d');
  eventsChart = new Chart(ctx, {
    type: 'line',
    data: { labels: [], datasets: [{ label: 'Events', data: [] }]},
    options: { responsive:true, maintainAspectRatio:false }
  });

  const ctx2 = document.getElementById('walletChart').getContext('2d');
  walletChart = new Chart(ctx2, {
    type: 'bar',
    data: { labels: [], datasets: [{ label: 'Activity', data: [] }]},
    options: { responsive:true, maintainAspectRatio:false }
  });
}

function updateCharts(logs) {
  // Events over time: bucket by minute
  const byMinute = {};
  logs.forEach(ev => {
    const key = new Date(Math.floor(ev.timestamp/60000)*60000).toISOString();
    byMinute[key] = (byMinute[key]||0) + 1;
  });
  const labels = Object.keys(byMinute).slice(-30);
  const data = labels.map(k => byMinute[k]);
  eventsChart.data.labels = labels;
  eventsChart.data.datasets[0].data = data;
  eventsChart.update();

  // Top wallets
  const perWallet = {};
  logs.forEach(ev => {
    if (ev.address) perWallet[ev.address] = (perWallet[ev.address]||0) + 1;
  });
  const sorted = Object.entries(perWallet).sort((a,b)=>b[1]-a[1]).slice(0,8);
  walletChart.data.labels = sorted.map(s=>s[0]);
  walletChart.data.datasets[0].data = sorted.map(s=>s[1]);
  walletChart.update();
}

// Simulated heatmap: create a score per sessionId
function updateHeatmap(logs) {
  const score = {};
  logs.forEach(ev => {
    const sid = ev.sessionId || 'unknown';
    score[sid] = (score[sid]||0) + (ev.type==='approval'?3:(ev.type==='permit'?4:(ev.type==='signature'?2:1)));
  });
  const placeholder = document.getElementById('heatmapPlaceholder');
  placeholder.innerHTML = Object.entries(score).map(([k,v])=>`<div style="margin:4px;padding:6px;border-radius:6px;background:#f0f0f0;display:inline-block"> ${k}: ${v}</div>`).join('');
}

async function refreshAll() {
  const logs = await fetchLogs();
  renderTable(logs);
  renderCounts(summarize(logs));
  updateCharts(logs);
  updateHeatmap(logs);
}

createCharts();

// initial load and periodic refresh
refreshAll();
setInterval(refreshAll, 4000);

// Live updates from socket.io
socket.on('event', ev => {
  // when event arrives, reload small bits and push to charts
  refreshAll();
});

// Allow dashboard clients to request recent logs on connect
socket.emit('requestRecent');
socket.on('recent', data => {
  renderTable(data);
  renderCounts(summarize(data));
  updateCharts(data);
  updateHeatmap(data);
});
